"""Reports page showing MIS dashboards and allowing exports."""

from __future__ import annotations

import io
import pandas as pd
import streamlit as st
import plotly.express as px

from services.report_service import ReportService


def render() -> None:
    st.title("Reports")
    report_service = ReportService()

    # --- Workload Distribution ---
    st.subheader("Workload Distribution")
    workload_df = report_service.workload_distribution()

    if workload_df.empty:
        st.info("No data available for workload distribution.")
    else:
        # Columns are: employee, tasks_assigned
        fig = px.bar(workload_df, x="employee", y="tasks_assigned", title=None)
        st.plotly_chart(fig, use_container_width=True)

        c1, c2 = st.columns(2)
        with c1:
            csv_bytes = workload_df.to_csv(index=False).encode("utf-8")
            st.download_button(
                "Download Workload CSV",
                data=csv_bytes,
                file_name="workload_distribution.csv",
                mime="text/csv",
                key="dl_workload_csv",
            )
        with c2:
            to_xlsx = io.BytesIO()
            with pd.ExcelWriter(to_xlsx, engine="openpyxl") as writer:
                workload_df.to_excel(writer, sheet_name="Workload", index=False)
            st.download_button(
                "Download Workload Excel",
                data=to_xlsx.getvalue(),
                file_name="workload_distribution.xlsx",
                mime="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
                key="dl_workload_xlsx",
            )

    # --- Overdue Assignments ---
    st.subheader("Overdue Assignments")
    delays_df = report_service.delay_report()
    if delays_df.empty:
        st.success("No overdue assignments.")
    else:
        st.dataframe(delays_df, use_container_width=True)
        c3, c4 = st.columns(2)
        with c3:
            csv_bytes = delays_df.to_csv(index=False).encode("utf-8")
            st.download_button(
                "Download Overdue CSV",
                data=csv_bytes,
                file_name="overdue_assignments.csv",
                mime="text/csv",
                key="dl_overdue_csv",
            )
        with c4:
            to_xlsx = io.BytesIO()
            with pd.ExcelWriter(to_xlsx, engine="openpyxl") as writer:
                delays_df.to_excel(writer, sheet_name="Overdue", index=False)
            st.download_button(
                "Download Overdue Excel",
                data=to_xlsx.getvalue(),
                file_name="overdue_assignments.xlsx",
                mime="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
                key="dl_overdue_xlsx",
            )

    # --- Compliance Calendar ---
    st.subheader("Compliance Calendar")
    cal_df = report_service.compliance_calendar()
    if cal_df.empty:
        st.info("No tasks in the calendar yet.")
    else:
        # Ensure dates display nicely
        if "due_date" in cal_df.columns:
            cal_df["due_date"] = pd.to_datetime(cal_df["due_date"], errors="coerce").dt.date.astype("object")
        st.dataframe(cal_df, use_container_width=True)

        csv_bytes = cal_df.to_csv(index=False).encode("utf-8")
        st.download_button(
            "Download Calendar CSV",
            data=csv_bytes,
            file_name="compliance_calendar.csv",
            mime="text/csv",
            key="dl_calendar_csv",
        )
